import { Modal, PrimaryButton } from '@/Components'
import StudentLayout from '@/Layouts/StudentLayout'
import { PageProps } from '@/types'
import { ID } from '@/types/models'
import { Level } from '@/types/models/level'
import { Material } from '@/types/models/material'
import { Student } from '@/types/models/student'
import { Link, useForm } from '@inertiajs/react'
import { Select } from 'antd'
import { FormEventHandler, useEffect, useState } from 'react'

type Props = {
    levels?: Level[],
    materials?: Material[],

}

const Home = ({ auth, levels, materials }: PageProps<Props>) => {


    if (levels && !auth.student?.level) {
        return <SelectLevel levels={levels} student={auth.student} />
    }

    return (
        <StudentLayout>
            <h1 className='text-3xl font-semibold text-white'>
                {auth.student.level?.title}
            </h1>
            <p className='text-md mt-5 text-white'>
                {auth.student.level?.description}
            </p>
            <div className="flex gap-5 mt-10">
                {auth.student.level?.materials?.map((item, i) => (
                    <Link href={route('student.material', { id: item.id })} key={i} className="bg-black border rounded-2xl h-[160px] w-[160px] flex justify-center items-center text-white flex-col hover:opacity-70 ">
                        <img src={item.cover_image} alt={item.title} />
                        <p className='mt-1 font-semibold'>
                            {item.title}
                        </p>
                    </Link>
                ))}
            </div>
        </StudentLayout>
    )
}

export default Home

const SelectLevel = ({ student, levels }: { student: Student, levels: Level[] }) => {
    const [modalOpen, setModalOpen] = useState(false);
    const [showSubLevel, setShowSubLevel] = useState(true);
    const [level, setLevel] = useState<Level>();
    const [subLevelId, setSubLevelId] = useState<ID>();

    const closeModal = () => {
        setModalOpen(false);
    };

    useEffect(() => {
        if (!student.level) {
            setModalOpen(true);
        }
    }, []);

    useEffect(() => {
        setShowSubLevel(Boolean(level?.sub_levels?.length)); // Show sublevel dropdown if sub_levels exist
        setSubLevelId(undefined); // Reset subLevelId when a new level is selected
    }, [level]);

    const { post } = useForm();

    const submit: FormEventHandler = (e) => {
        e.preventDefault();
        post(route('student.addLevel', { id: level?.id, sub: subLevelId, }), {
            onSuccess: () => closeModal(),
        });
    };

    return (
        <Modal show={modalOpen} onClose={closeModal}>
            <form onSubmit={submit} className="flex flex-col items-center text-black mb-5">
                <h2 className="text-2xl mb-10">إختر الصف الخاص بك</h2>
                <Select className="w-1/2 h-[40px]" onSelect={(value) => setLevel(levels.find((l) => l.id === value))}>
                    {levels?.map((level) => (
                        <Select.Option key={level.id} value={level.id}>
                            {level.title}
                        </Select.Option>
                    ))}
                </Select>

                {showSubLevel && (
                    <Select className="w-1/2 h-[40px] mt-5" onSelect={(value) => setSubLevelId(value as ID)}>
                        {level?.sub_levels?.map((subLevel) => (
                            <Select.Option key={subLevel.id} value={subLevel.id}>
                                {subLevel.title}
                            </Select.Option>
                        ))}
                    </Select>
                )}

                <PrimaryButton className="w-1/2 mt-5">حفظ</PrimaryButton>
            </form>
        </Modal>
    );
};
