import { useState, useRef, useEffect } from "react";
import axios from "axios";
import { Upload } from "lucide-react";
import { API_URL } from "@/Constants";


type ApiResponseProps = {
    file_path: string;
}

interface FormUploadProps {
  type: 'image' | 'video' | 'document' | 'audio';
  className?: string;
  onSuccess?: (response: ApiResponseProps) => void; // onSuccess prop to return file path
  apiPath?: string
}

const FormUpload: React.FC<FormUploadProps> = ({ type, className, onSuccess, apiPath }) => {
  const [uploading, setUploading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<boolean>(false);
  const [file, setFile] = useState<File | null>(null);

  const fileInputRef = useRef<HTMLInputElement | null>(null);

  // Define acceptable file types based on the `type` prop
  const fileAccept = (): string => {
    switch (type) {
      case "image":
        return "image/*";
      case "audio":
        return "audio/*";
      case "video":
        return "video/*";
      case "document":
        return ".pdf,.doc,.docx,.txt,.xls,.xlsx,.json";
      default:
        return "*/*"; // Accept all file types if no match
    }
  };

  const handleFileUpload = async (selectedFile: File) => {
    setUploading(true);
    setError(null);

    const formData = new FormData();
    formData.append("file", selectedFile);

    try {

      const PATH = apiPath ? apiPath : `/upload/${type}`

      const response = await axios.post(API_URL + PATH, formData, {
        headers: {
          "Content-Type": "multipart/form-data",
        },
      });

      setSuccess(true);

      // If the upload is successful, call onSuccess with the file path
      if (onSuccess) {
        onSuccess(response.data);
      }
    } catch (err: any) {
      setError(err.response?.data?.message || "File upload failed");
      console.error(err)
    } finally {
      setUploading(false);
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      setFile(e.target.files[0]);
      handleFileUpload(e.target.files[0]);
    }
  };

  const handleDrop = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    const droppedFiles = e.dataTransfer.files;
    if (droppedFiles && droppedFiles[0]) {
      setFile(droppedFiles[0]);
      handleFileUpload(droppedFiles[0]);
    }
  };

  const handleDragOver = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
  };



  return (
    <div className="space-y-5">
      <div
        className={`bg-gray-100 border-2 border-dashed border-gray-400 p-4 rounded-lg h-52 flex flex-col gap-5 items-center justify-center ${className}`}
        onDrop={handleDrop}
        onDragOver={handleDragOver}
        onClick={() => fileInputRef.current?.click()}
      >
        <Upload className="h-8 w-8" />
        <input
          type="file"
          accept={fileAccept()}
          onChange={handleFileChange}
          className="hidden"
          ref={fileInputRef}
        />
        <div className="text-center">
          {uploading ? (
            <p className="text-blue-500">Uploading...</p>
          ) : file ? (
            <p>{file.name}</p>
          ) : (
            <p className="text-gray-600 font-semibold" >Drag & drop a file here, or click to select a file</p>
          )}
        </div>

        {error && <p className="text-red-500 mt-2">{error}</p>}
        {success && <p className="text-green-500 mt-2">File uploaded successfully!</p>}
      </div>
    </div>
  );
};

export default FormUpload;

